from knot_resolver.datamodel.forward_schema import ForwardServerSchema
from knot_resolver.datamodel.templates import template_from_str


def test_boolean():
    tmpl_str = """{% from 'macros/common_macros.lua.j2' import boolean %}
{{ boolean(x) }}"""

    tmpl = template_from_str(tmpl_str)
    assert tmpl.render(x=True) == "true"
    assert tmpl.render(x=False) == "false"


def test_boolean_neg():
    tmpl_str = """{% from 'macros/common_macros.lua.j2' import boolean %}
{{ boolean(x,true) }}"""

    tmpl = template_from_str(tmpl_str)
    assert tmpl.render(x=True) == "false"
    assert tmpl.render(x=False) == "true"


def test_string_table():
    s = "any string"
    t = [s, "other string"]
    tmpl_str = """{% from 'macros/common_macros.lua.j2' import string_table %}
{{ string_table(x) }}"""

    tmpl = template_from_str(tmpl_str)
    assert tmpl.render(x=s) == f"'{s}'"
    assert tmpl.render(x=t) == f"{{'{s}','{t[1]}',}}"


def test_str2ip_table():
    s = "2001:DB8::d0c"
    t = [s, "192.0.2.1"]
    tmpl_str = """{% from 'macros/common_macros.lua.j2' import str2ip_table %}
{{ str2ip_table(x) }}"""

    tmpl = template_from_str(tmpl_str)
    assert tmpl.render(x=s) == f"kres.str2ip('{s}')"
    assert tmpl.render(x=t) == f"{{kres.str2ip('{s}'),kres.str2ip('{t[1]}'),}}"


def test_qtype_table():
    s = "AAAA"
    t = [s, "TXT"]
    tmpl_str = """{% from 'macros/common_macros.lua.j2' import qtype_table %}
{{ qtype_table(x) }}"""

    tmpl = template_from_str(tmpl_str)
    assert tmpl.render(x=s) == f"kres.type.{s}"
    assert tmpl.render(x=t) == f"{{kres.type.{s},kres.type.{t[1]},}}"


def test_servers_table():
    s = "2001:DB8::d0c"
    t = [s, "192.0.2.1"]
    tmpl_str = """{% from 'macros/common_macros.lua.j2' import servers_table %}
{{ servers_table(x) }}"""

    tmpl = template_from_str(tmpl_str)
    assert tmpl.render(x=s) == f"'{s}'"
    assert tmpl.render(x=t) == f"{{'{s}','{t[1]}',}}"
    assert tmpl.render(x=[{"address": s}, {"address": t[1]}]) == f"{{'{s}','{t[1]}',}}"


def test_tls_servers_table():
    d = ForwardServerSchema(
        # the ca-file is a dummy, because it's existence is checked
        {"address": ["2001:DB8::d0c"], "hostname": "res.example.com", "ca-file": "/etc/passwd"}
    )
    t = [
        d,
        ForwardServerSchema(
            {
                "address": "192.0.2.1",
                "pin-sha256": "E9CZ9INDbd+2eRQozYqqbQ2yXLVKB9+xcprMF+44U1g=",
            }
        ),
    ]
    tmpl_str = """{% from 'macros/common_macros.lua.j2' import tls_servers_table %}
{{ tls_servers_table(x) }}"""

    tmpl = template_from_str(tmpl_str)
    assert tmpl.render(x=[d.address, t[1].address]) == f"{{'{d.address}','{t[1].address}',}}"
    assert (
        tmpl.render(x=t)
        == f"{{{{'{d.address}',hostname='{d.hostname}',ca_file='{d.ca_file}',}},{{'{t[1].address}',pin_sha256={{'{t[1].pin_sha256}',}}}},}}"
    )
